/*
 * A simple implementation of the World magnetic model 
 * (https://www.ngdc.noaa.gov/geomag/WMM/) 
 *
 * (c) 2021 Jiri Pittner <jiri@pittnerovi.com>
 *
 * This implementation is simplified by omiting the geoid to WGS84 ellipsoid altitude correction
 * since the geoid data is inconveniently large for use in microcontrollers
 * and this approximation is good enough for practical purposes in simple avionics
 * My implementation should also be much more compact and lucid than the official one
 * and avoids any file I/O, as it is primarily intended for embedded devices
 * The WMM coefficients are located in wmm_data.h and can be generated from the 
 * official distribution file by a simple script (see the header file for details)
 *
 *
 *       This program is free software: you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation, either version 3 of the License, or
 *      (at your option) any later version.
 *
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 *      You should have received a copy of the GNU General Public License
 *      along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 */

#include <stdlib.h>
#include <math.h>
#include "wmm.h"
#include "wmm_data.h"

//support different reals
#if ( WMM_REAL == double )
#define wmm_sqrt sqrt
#define wmm_sin sin
#define wmm_cos cos
#define wmm_tan tan
#define wmm_asin asin
#define wmm_acos acos
#define wmm_atan2 atan2
#else
#define wmm_sqrt sqrtf
#define wmm_sin sinf
#define wmm_cos cosf
#define wmm_tan tanf
#define wmm_asin asinf
#define wmm_acos acosf
#define wmm_atan2 atan2f
#endif



//auxiliary math functions

WMM_REAL Plm(int l, int m, WMM_REAL x)
{
        WMM_REAL fact,pll,pmm,pmmp1,somx2;
        int i,ll;

        if (m < 0 || m > l || x > 1. || x < -1.) return NAN;
        pmm=1.0;
        if (m > 0) {
                somx2 = wmm_sqrt((1.0-x)*(1.0+x));
                fact = 1.0;
                for (i=1;i<=m;i++) {
                        pmm *= -fact*somx2;
                        fact += 2.0;
                }
        }
        if (l == m)
                return pmm;
        else {
                pmmp1 = x*(2*m+1)*pmm;
                if (l == (m+1))
                        return pmmp1;
                else {
                        for (ll=m+2;ll<=l;ll++) {
                                pll=(x*(2*ll-1)*pmmp1-(ll+m-1)*pmm)/(ll-m);
                                pmm=pmmp1;
                                pmmp1=pll;
                        }
                        return pll;
                }
        }
}

#define MAXFACT (2*MAXL+1)
WMM_REAL fact(int n)
{
int j;
static int ntop=20;
static WMM_REAL a[MAXFACT+1]={1.0,1.0,2.0,6.0,24.0,120.0,720.0,5040.0,
40320.0,
362880.0,
3628800.0,
39916800.0,
479001600.0,
6227020800.0,
87178291200.0,
1307674368000.0,
20922789888000.0,
355687428096000.0,
6402373705728000.0,
121645100408832000.0,
2432902008176640000.0};

if (n < 0 || n>MAXFACT) return NAN;
while (ntop<n) {
                j=ntop++;
                a[ntop]=a[j]*ntop;
                }
return a[n];
}


WMM_REAL legendre(int l, int m, WMM_REAL x)
{
WMM_REAL r =  Plm(l,m,x);
if(m==0) return r;
if(m&1) r= -r; //different sign convention 
WMM_REAL norm = wmm_sqrt(2*fact(l-m)/fact(l+m)); //different normalization
return r*norm;
}


//WGS84 ellipsoid parameters
const WMM_REAL A = 6378137.; //major axis
//flattening
#define F  (1./298.257223536) 
const WMM_REAL e2 = F*(2.-F); //eccentricity


//WMM calculation - see file WMM2020_Report.pdf for description and equations
//time is in years, altitudes in meters, coordinates in degrees
int wmm(WMM_REAL lattitude, WMM_REAL longitude, WMM_REAL altitude, WMM_REAL time,  WMM_REAL *x, WMM_REAL *y, WMM_REAL *z, WMM_REAL *declination, WMM_REAL *inclination)
{
//convert from WGS84 ellipsoid geodetic coordinates to spherical geocentric coordinates
WMM_REAL lambda = longitude*M_PI/180.;
WMM_REAL phi = lattitude*M_PI/180.;
WMM_REAL s = wmm_sin(phi);
WMM_REAL Rc = A/wmm_sqrt(1-e2*s*s);
WMM_REAL p = (Rc + altitude) * wmm_cos(phi);
WMM_REAL zz = s*(Rc*(1.-e2)+altitude);
WMM_REAL r=wmm_sqrt(p*p+zz*zz);
WMM_REAL sinphiprime = zz/r;
WMM_REAL cosphiprime = wmm_sqrt(1.-sinphiprime*sinphiprime);
WMM_REAL tanphiprime = sinphiprime/cosphiprime;
WMM_REAL phiprime = wmm_asin(sinphiprime);

int n,m;

//precompute sines and cosines to improve efficiency
WMM_REAL sinml[MAXL+1];
WMM_REAL cosml[MAXL+1];
sinml[0]=0.;
cosml[0]=1.;
{
WMM_REAL ml=0;
for(m=1; m<=MAXL; ++m)
	{
	ml+= lambda;
	sinml[m] = wmm_sin(ml);
	cosml[m] = wmm_cos(ml);
	}
}


//compute magnetic field in  spherical geocentric coordinates
const WMM_REAL a = 6371200.;
WMM_REAL xprime=0;
WMM_REAL yprime=0;
WMM_REAL zprime=0;
WMM_REAL a_over_r_power = a*a/(r*r);
int ithru=0;
for(n=1; n<=MAXL; ++n)
	{
	a_over_r_power *= (a/r);
	WMM_REAL xsum=0;
	WMM_REAL ysum=0;
	WMM_REAL zsum=0;
	for(m=0; m<=n; ++m)
		{
		//check that order in the coefficient file matches 
		if(wmmdata[ithru].n!=n || wmmdata[ithru].m!=m) 
			{
			*x = *y = *z = *inclination = *declination = 0; //safe error handling
			return 1;
			}
		//interpolate expansion coefficients, 
		WMM_REAL gt = wmmdata[ithru].g + (time-START_YEAR)* wmmdata[ithru].gder;
		WMM_REAL ht = wmmdata[ithru].h + (time-START_YEAR)* wmmdata[ithru].hder;

		//compute summands
		WMM_REAL leg = legendre(n,m,sinphiprime);
		WMM_REAL legendre_der = (n+1)*tanphiprime*leg - wmm_sqrt((n+1)*(n+1)-m*m)/cosphiprime*legendre(n+1,m,sinphiprime);
		WMM_REAL tmp = gt*cosml[m] + ht * sinml[m];
		xsum += tmp * legendre_der;
		ysum += m*(gt*sinml[m] - ht * cosml[m]) * leg;
		zsum += tmp * leg;
		
		++ithru; //next coefficient entry
		}
	xprime += a_over_r_power * xsum;
	yprime += a_over_r_power * ysum;
	zprime += (n+1) * a_over_r_power * zsum;
	}
//pre-sum factors
xprime = -xprime;
yprime /= cosphiprime;
zprime = -zprime;

//convert magnetic field back to wgs84
WMM_REAL sindif = wmm_sin(phiprime-phi);
WMM_REAL cosdif = wmm_cos(phiprime-phi);
*x = xprime * cosdif - zprime * sindif;
*y = yprime;
*z = xprime * sindif + zprime * cosdif;

//compute declination and inclination
WMM_REAL tmp = *x * *x + *y * *y;
WMM_REAL h = wmm_sqrt(tmp);
//WMM_REAL f = wmm_sqrt(tmp + *z * *z);
*inclination = wmm_atan2(*z,h) *180./M_PI;
*declination = wmm_atan2(*y,*x) *180./M_PI;

if(time<START_YEAR || time > END_YEAR) return 2; //warning - outdated model

return 0;
}

//time conversion
WMM_REAL time2year(const time_t t)
{
struct tm tmp;
tmp.tm_sec=0;
tmp.tm_min=0;
tmp.tm_hour=0;
tmp.tm_mday=1;
tmp.tm_mon=0;
tmp.tm_isdst=0;
tmp.tm_year=START_YEAR-1900;
time_t t0=mktime(&tmp);
return START_YEAR+(t-t0)/(3600*24*365.25);
}


//testing facility

#ifdef STANDALONE
#include <stdio.h>
int main()
{
double lattitude,longitude,altitude,t,x,y,z,d,i;
printf("Enter lattitude longitude (degrees) altitude (meters) time (year): ");
scanf("%lf %lf %lf %lf",&lattitude,&longitude, &altitude, &t);
if(t==0) 
	{
	t=time2year(time(NULL));
	printf("current time is %f\n",t);
	}
int r=wmm(lattitude,longitude,altitude,t,&x,&y,&z,&d,&i);
if(r) printf("calculation error %d\n",r);
else printf("\nxyz %f %f %f (nanotesla) declination= %f inclination= %f (degrees)\n",x,y,z,d,i);
return 0;
}
#endif
