/*
 * 	termostat3.c ... firmware for the open hardware thermostat
 *      corresponds to PCB meteo3.sch v. 1.3 and higher combining meteostation and thermostat
 *
 * 	Copyright (C) 2010-2014 Jiri Pittner <jiri@pittnerovi.com>
 *
 * 	This program is free software: you can redistribute it and/or modify
 * 	it under the terms of the GNU General Public License as published by
 * 	the Free Software Foundation, either version 3 of the License, or
 * 	(at your option) any later version.
 *
 * 	This program is distributed in the hope that it will be useful,
 * 	but WITHOUT ANY WARRANTY; without even the implied warranty of
 * 	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * 	GNU General Public License for more details.
 * 	You should have received a copy of the GNU General Public License
 * 	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 	                                         
 * 	                                        
*/

#undef debug
#undef debug2

/* TODO:
 * @@@kontrola watchdogovani - v plzni se mozna jeden zasekl
 */

#define LOCATION chatagaraz
#define SENSOR_SERIAL 6
#define ACTOR_SERIAL 9

//needed with the newer smt172 sensor
//#define EXTRA_AVERAGE

//NOTE: DS18B20 has a different pinout than SMT160!!!
//NOTE: internal pullup might be too weak, 4k7 pullup recommended for DS18B20


//update also temp_timeouts_thr[NTHERMOMETERS] if increasing NTHERMOMETERS

//hardware - cf. Kicad schematics meteo3.pro - PCB version 1.3 combined to build either meteostation or thermostat
//atmega128, 14.7456, upload also eeprom!
//DS18B20 needs extra 4k7 pullup on dfata pin

#define COUNTER_OUTPUT_INIT  5000
#define COUNTER_SENSOR1_INIT  0
#define COUNTER_SENSOR2_INIT  0
#define COUNTER_SENSOR3_INIT 0
#define COUNTER_ACTOR0_INIT 6
#define COUNTER_ACTOR1_INIT 0
#define COUNTER_ACTOR2_INIT 0
#define COUNTER_ACTOR3_INIT 0
#define COUNTER_ACTOR4_INIT 0
#define COUNTER_ACTOR5_INIT 0
#define COUNTER_ACTOR6_INIT 0
#define COUNTER_ACTOR7_INIT 1



#define COUNTER32

#define BAUD 115200

/*PIN assignment
 *

433 modules:
	PD0 IN rx433
	PB5 OUT tx433
	PB7 OUT tx_on 

LCD display:
	PC0-3=DB4-7
	PC4=E
	PC5=RW
	PC6=RS 
	PC7=light 

rotational coder 
	PE6 (left)
	PE7 (right)
	PE4 (press)

builtin relay output: PE3.

ISP : PE0,1 + SCK + reset 

temperature:
	PE2 (SMT160-30) - on raingauge connector, not SMT2

relays:
	PA0 heating 
	PA1 cooling 
	PA2 heater power relay 
	PA3 heating2
LEDs:
	PA4 red 
	PA5 green
	PA6 blue
	PA7 reserved


*/ 

#define TEMPPORT PINE
#define TEMPDDR DDRE
#define TEMPPORTOUT PORTE
#define TEMPPIN PE2

#define MEASUREMODULO 0x0f
#define AVERAGING 8
#define TXCOUNT 5

#include "backward.h"
#include <avr/io.h>
#include <avr/pgmspace.h>
#include <avr/eeprom.h>
#include <avr/interrupt.h>
#include <setjmp.h>
#include <avr/sleep.h>
//obsolete #include <avr/timer.h>
#include <avr/wdt.h>
#include <stdlib.h>
#include <string.h>

#define INCFILE_PARAMS_(X) X##_params.h
#define INCFILE_PARAMS(X) INCFILE_PARAMS_(X)
#define INCFILE_KEYS_(X) X##_keys.h
#define INCFILE_KEYS(X) INCFILE_KEYS_(X)
#define XSTR(s) STR(s)
#define STR(s) #s

#define STRLOCATION STR(LOCATION)

#include XSTR(INCFILE_PARAMS(LOCATION))
#include "keeloq2.h"
#include XSTR(INCFILE_KEYS(LOCATION))
#include "devices433.h"


//how to force this to start at address 0 - by giving it an initialization
//activate this to shift the eeprom contents if needed
/*
static volatile uint8_t ee_wear_leveling[0x0] __attribute__ ((used,section(".eeprom"))) =
{
}; //use now fresh eeprom places for the real variables
*/


#if defined(at90s2313) || defined(at90s8535)
#else
#define ATmega
#endif

#ifdef ATmega
#define USR UCSRA
#endif


#if (XTAL == 14745600L) 
#define oversampling 4
#else
#define oversampling 2
#endif

#ifndef MCU
#define emulate
#else
#define AVR_assembler
#endif

#define itoa10(N,S) itoa(N,S,10)
#define itoa16(N,S) itoa(N,S,16)


void printP (PGM_P string){
        char c;
        c=pgm_read_byte(string);
                   while (c) {
                   loop_until_bit_is_set(UCSR1A, UDRE);
                   UDR1 = c;
                   c=pgm_read_byte(++string);
                   }
                   return;
                 }



void print (char *string){                                     
                   while (*string) {                                            
                   loop_until_bit_is_set(UCSR1A, UDRE);                            
                   UDR1 = *string++;
                   }                                                            
                   return;                                                      
                 }                                                              


void bin2letter(char *c, uint8_t b)
{
*c = b<10? '0'+b : 'A'+b-10;
}

void bin2octet(char *octet, uint8_t bin)
{
bin2letter(octet,bin>>4);
bin2letter(octet+1,bin&0x0f);
}





//UART initialize
#define UART_INIT(baud) { \
UBRR1H=0; \
UBRR1L= (XTAL/baud+15)/16-1; \
UCSR1B=(1<<TXEN)|(1<<RXEN); \
UCSR1C=(1<<UCSZ0)|(1<<UCSZ1)|(1<<USBS); }



//UART routines
volatile uint8_t inuartlen=0;
volatile uint8_t uartline=0;
#define MAXUART 64
volatile char inuart[MAXUART];
INTERRUPT(SIG_UART1_RECV)
{
char c=UDR1;
if(uartline) return; //ignore until line is processed
UDR1 = c; //echo
if(c=='\n' || c== '\r' || inuartlen == MAXUART-1) {uartline=1; inuart[inuartlen]=0; inuartlen=0;} 
else inuart[inuartlen++]=c; 
}



watchdog(uint8_t onoff)
{
if(onoff) {wdt_enable(WDTO_2S); wdt_reset();}
else {wdt_reset();wdt_disable();}
}


#include "keeloq2.c"

//transmit routines

volatile uint8_t outcode[MESSAGE_BYTES];
volatile uint8_t outcodelen=0; //0 indicates ready to receive, non-zero indicates ready to read it in main
volatile uint8_t waittime=0;
volatile uint8_t txcount=0; //repeat several times
volatile static uint8_t obitcount=0, obytecount=0; //working counters

typedef enum {init,oupmark,downmark,data,trailer,wait} OSTATE;
static OSTATE ostate=init;
static uint8_t ocount=0;




void txcar(void)
{
switch (ostate) {
	case wait: if(--waittime) return; 
        case init: //the user has just written the request to transmit
			//check maxcodelen
			ocount=13;
			ostate=oupmark;
			cbi(TIMSK,TOIE1);
			ICR1=(uint16_t)(XTAL*8./10000+.5)-1;
                	OCR1A=((uint16_t)(XTAL*4./10000+.5)-1);
			TCNT1=0;
			cbi(TCCR1A,COM1A0);
			cbi(TCCR1A,COM1A1);
			sbi(TIMSK,TOIE1);
			sbi(TIMSK,OCIE1A);
                break;
        case oupmark:
		if(--ocount) {sbi(PORTB,PB5); return;}
		cbi(PORTB,PB5);
		cbi(TIMSK,TOIE1);
                TCNT1=0;
		ICR1=(uint16_t)(XTAL*6.1/10000+.5)-1;
                sbi(TIMSK,TOIE1);
		ostate=downmark;
		ocount=2;
                break;
        case downmark:
		if(--ocount) return;
		obitcount=obytecount=0;
		ostate=data;
		cbi(TIMSK,TOIE1);
		ICR1=(uint16_t)(XTAL*12./10000+.5)-1;
                sbi(TIMSK,TOIE1);
		break;
        case data:
		{
		uint8_t n;
		n=obitcount+(obytecount<<3);
		if(n>outcodelen+1)
                        {
                        //data finished 
                cbi(PORTB,PB5);
			cbi(TIMSK,OCIE1A);
			ostate=trailer;
			ocount=13;
			OCR1A=0;
                        }
		else
			{
                        if(n) sbi(PORTB,PB5); //avoid spurious first peak
			if(n>outcodelen) 
				{
				OCR1A=((uint16_t)(XTAL*12./10000+.5)-1)*1/3;
				++obitcount;
				}
			else
				{
				OCR1A=((uint16_t)(XTAL*12./10000+.5)-1)*
				((outcode[obytecount]& (1<<obitcount))?1:2)/3;
				if(++obitcount==8) {obitcount=0;++obytecount;}
				}
			}
		}
                break;
        case trailer:
		if(--ocount) return;
		//finished
                if(--txcount == 0) {ostate=init; outcodelen=0;}
                else {ostate=wait; waittime=32;} 
                break;
}
}



INTERRUPT(SIG_OVERFLOW1)
{
if(!outcodelen) return;
txcar();
}


INTERRUPT(SIG_OUTPUT_COMPARE1A)
{
cbi(PORTB,PB5); 
}

//receive routines
uint8_t incode2[MESSAGE_BYTES+1];
uint8_t incodelen2=255;
volatile uint8_t incode[MESSAGE_BYTES+1];
volatile uint8_t incodelen=0; //0 indicates ready to receive, non-zero indicates ready to read it in main
volatile static uint8_t bitcount=0, bytecount=0; //working counters

typedef enum {noise,upmark,ready} STATE;

//receive routine for the code-hopping car remote - PWM coded

void receive_car2(uint16_t time1, uint16_t time0)
{
static STATE state=noise;
static uint8_t headercount=0;
//{char c[6]; print(itoa(time1,c,10)); print(":"); print(itoa(time0,c,10)); print("\n"); }
switch(state) {
	case noise: 
		if(abs(time1-time0)<=2*oversampling) headercount++; else headercount=0;
		if(headercount>6) state=upmark;
		break;
	case upmark:
		if(time1>=3*oversampling && time1<=6*oversampling && time0>=30*oversampling&& time0<=60*oversampling) {state=ready; incode[0]=bitcount=bytecount=0;break;}
		if(time1>=3*oversampling && time1<=6*oversampling && time0>=3*oversampling && time0<=6*oversampling) headercount++; else {state=noise; headercount=bitcount=bytecount=0;}
		break;
	case ready:
		//{char c[6]; print(itoa(time1,c,10)); print(":"); print(itoa(time0,c,10)); print("\n"); }
		if(time1==time0||time1<2*oversampling || time1 >12*oversampling || time0<2*oversampling) {state=noise; incode[0]=incodelen=bitcount=bytecount=0; break;} //unexpected abort
		incode[bytecount]|= (time1>time0?0:1)<<bitcount++;
		if(bitcount==8) {bitcount=incode[++bytecount]=0; }
		if(time0>60*oversampling || bytecount>=MESSAGE_BYTES+1) {state=noise; incodelen=bitcount+(bytecount<<3)-1; } //legal end of transmition
		break;
	}
}


void receive_car(uint8_t level,uint16_t time)
{
static uint16_t time1=0;
//{char c[6]; print(level?"1:":"0:"); print(itoa(time,c,10)); print("\n");}
if(level) time1=time; else receive_car2(time1,time);
}



static uint16_t count[2]={0,0};
static uint8_t laststate=0;

//this generates a run length encoded stream of data from the receiver and passes control to specialized decoders
//but it seems to work reasonably like this
INTERRUPT(SIG_OUTPUT_COMPARE1B)
{
uint8_t state;
TCNT1=0; //with B it cannot run in CTC mode
state=(PIND&0x01)?1:0;
if(state==laststate) {if(++count[state]<=100*oversampling) return;}
//state!=laststate or same state for too long
	{
	uint8_t l;
	uint16_t t;
	t=count[l=laststate];
	laststate=state;
	count[state]=1;
	count[l]=0;
	if(incodelen) return; //buffer is full, main has to read it - ignore new data
	receive_car(l,t);
	}
}




//rotational coder routines
volatile static uint8_t lock_tempset;
volatile static int16_t tempset;
volatile static uint8_t hysteresis_on;
volatile static uint8_t hysteresis_off;
#define TEMPSET_DEFAULT 220
static int16_t ee_tempset __attribute__ ((section(".eeprom"))) = TEMPSET_DEFAULT;
#define HYSTERESIS_ON_DEFAULT 5
static uint8_t ee_hysteresis_on __attribute__ ((section(".eeprom"))) = HYSTERESIS_ON_DEFAULT;
#define HYSTERESIS_OFF_DEFAULT 1
static uint8_t ee_hysteresis_off __attribute__ ((section(".eeprom"))) =HYSTERESIS_OFF_DEFAULT;
#define TEMPMIN_INIT 1000
static int16_t ee_tempmin __attribute__ ((section(".eeprom"))) = TEMPMIN_INIT;
#define TEMPMAX_INIT -1000
static int16_t ee_tempmax __attribute__ ((section(".eeprom"))) = TEMPMAX_INIT;
static int32_t tempsum = 0; //!NOT eeprom
static uint16_t tempcount = 0; //!NOT eeprom
static uint16_t ee_nreset  __attribute__ ((section(".eeprom"))) = 0;

volatile static uint8_t timer0x=0;
volatile static uint8_t menustate=0;
volatile static int8_t menuselect=0;

volatile static uint8_t tempselect=0;
volatile static uint8_t meteoselect=0;
static volatile uint8_t eff_temp_from;
static volatile uint8_t eff_temp_to;
static volatile uint8_t eff_temp_kind;


#define N_MENULABEL 16
static const MENULABEL  menulabel[N_MENULABEL] = {"FUNCTION","HYSTON","HYSTOFF","MINIMUM","AVERAGE","MAXIMUM","TEMP ALL","TEMPFROM","TEMP TO","TEMPKIND","RESET H!","# RESETS","METEO   ","LOCK    ","HEATWATR","TAPWATER"};

static const char label[5]="OHCXR";
volatile static uint8_t activestate;
#define ACTIVESTATE_DEFAULT 1
static uint8_t ee_activestate __attribute__ ((section(".eeprom"))) = ACTIVESTATE_DEFAULT;
volatile static uint16_t clearingcounter=0;

volatile uint8_t heatwater=0; //AUTO 
volatile uint8_t tapwater=50;


void process_rotation(uint8_t sense)
{
clearingcounter=0;
switch (menustate) {
        case 16: //heat water temp
		if(sense) ++heatwater; else {if(heatwater>0) --heatwater;}
                if(heatwater<40) heatwater=0; //AUTO
                if(heatwater>75) heatwater=75;
                break;
        case 17: //tap water temp
                if(sense) ++tapwater; else --tapwater;
                if(tapwater<15) tapwater=15;
                if(tapwater>60) tapwater=60;
                break;
	case 12://reset heater
		break;
	case 13://show no. of resets
		break;
	case 14://meteo
		if(sense) ++meteoselect; else {if(meteoselect>0) --meteoselect;}
		if(meteoselect>6) meteoselect=6;
		break;
	case 15:
		lock_tempset ^= 1; //regardless of rotation sense
		break;
	case 8:
		if(sense) ++tempselect; else {if(tempselect>0) --tempselect;}
		if(tempselect>=NTHERMOMETERS) tempselect=NTHERMOMETERS-1;
		break;
	case 9:
		if(sense) ++eff_temp_from; else {if(eff_temp_from>0) --eff_temp_from;}
		if(eff_temp_from>=NTHERMOMETERS) eff_temp_from=NTHERMOMETERS-1;
		break;
	case 10:
		if(sense) ++eff_temp_to; else {if(eff_temp_to>0) --eff_temp_to;}
		if(eff_temp_to<eff_temp_from) eff_temp_to=eff_temp_from;
		if(eff_temp_to>=NTHERMOMETERS) eff_temp_to=NTHERMOMETERS-1;
		break;
	case 11:
		if(sense) ++eff_temp_kind;  else {if(eff_temp_kind>0) --eff_temp_kind;}
		if(eff_temp_kind>MAXIMUM) eff_temp_kind=MAXIMUM;
		break;
        case 7:
	case 6:
	case 5:
		break;
        case 4:
		if(sense) ++hysteresis_off; else {if(hysteresis_off>0) --hysteresis_off;}
		if(hysteresis_off> 99) hysteresis_off=99;
		if(hysteresis_off >= hysteresis_on) hysteresis_off=hysteresis_on-1;
                break;
        case 3:
		if(sense) ++hysteresis_on; else {if(hysteresis_on>0) --hysteresis_on;}
		if(hysteresis_on > 99) hysteresis_on=99;
		if(hysteresis_on < 1) hysteresis_on=1;
		if(hysteresis_off >= hysteresis_on) hysteresis_off=hysteresis_on-1;
                break;
        case 2:
		if(sense) ++activestate; else --activestate;
		activestate &= 3;
                break;
        case 1:
		if(sense) ++menuselect; else --menuselect;
		if(menuselect<0) menuselect+=N_MENULABEL;
		menuselect = menuselect%N_MENULABEL;
                break;
        case 0:
        default:
		if(lock_tempset) break; //ignore if locked
		if(sense) {++tempset; if(tempset>999) tempset=999;} else {--tempset; if(tempset<-999) tempset=-999;}
		break;
}
}


//right:
//I0:01
//I1:00
//I0:10
//I1:11
//
//left:
//I1:10
//I0:00
//I1:01
//I0:11
//
volatile static int8_t rotcount=0;
void rotcoder(uint8_t interrupt, uint8_t pins)
{
if(interrupt)
	{
	if(pins==0 || pins==3) ++rotcount;
	else --rotcount;
	}
else
	{
	if(pins==0 || pins==3) --rotcount;
	else ++rotcount;
	}
if(rotcount==4) {process_rotation(1); rotcount=0;}
if(rotcount== -4) {process_rotation(0); rotcount=0;}
}

volatile static uint8_t light=0;
volatile static uint8_t reset_heater_pending;

void process_press(uint8_t length)
{
clearingcounter=0;
switch (menustate) {
	case 12: //reset heater
		if(length) reset_heater_pending=1; else menustate=1;
                break;
	case 7:
		if(length) eeprom_write_word(&ee_tempmax, -1000);
		else menustate=1;
		break;
	case 6:
		if(length)
			{
			tempsum=0;
			tempcount=0;
			}
		else menustate=1;
		break;
	case 5:
		if(length) eeprom_write_word(&ee_tempmin, 1000);
		else menustate=1;
		break;
        case 16: //heatwater
                {
                CODE433 c;
                c.cmd.instr= THERMOSTAT|SET_HEATERTEMP;
                c.cmd.arg1= 0;
                c.cmd.arg2= 10*heatwater;
#ifdef HEATER_ACTOR
                transmit_actor(HEATER_ACTOR,c.code,TXCOUNT);
#endif
                }
                menustate=1;
                break;
        case 17: //tapwater
                {
                CODE433 c;
                c.cmd.instr= THERMOSTAT|SET_BOILERTEMP;
                c.cmd.arg1= 0;
                c.cmd.arg2= 10*tapwater;
#ifdef HEATER_ACTOR
                transmit_actor(HEATER_ACTOR,c.code,TXCOUNT);
#endif
                }
                menustate=1;
                break;
	case 13: //show no. of resets
	case 14: //meteo
	case 15: //lock
	case 8:
	case 9:
	case 10:
	case 11:
	case 4:
	case 3:
	case 2:
		menustate=1;
		break;
	case 1:
		if(length)
			{
			menustate=0;
			}
		else
			{
			menustate=menuselect+2;
	 		}
		break;
	case 0:
	default:
		if(length)  //enter menu by long press
			{
			menustate=1;
			menuselect=0;
			}
		else	   //toggle light
			{ 
			light ^= (1<<PC7);
			PORTC ^= (1<<PC7);
			}
}
}

INTERRUPT(SIG_INTERRUPT6)
{
rotcoder(0,(PINE>>6)&3);
}

INTERRUPT(SIG_INTERRUPT7)
{
rotcoder(1,(PINE>>6)&3);
}


static uint16_t counter_eeprom_output __attribute__ ((section(".eeprom"))) = COUNTER_OUTPUT_INIT;
static uint32_t counters_stored[NACTORS] __attribute__ ((section(".eeprom"))) = {COUNTER_ACTOR0_INIT,COUNTER_ACTOR1_INIT,COUNTER_ACTOR2_INIT,COUNTER_ACTOR3_INIT,COUNTER_ACTOR4_INIT,COUNTER_ACTOR5_INIT,COUNTER_ACTOR6_INIT,COUNTER_ACTOR7_INIT};
static volatile uint16_t counter_output= COUNTER_OUTPUT_INIT ; 

INTERRUPT(SIG_INTERRUPT4)
{
uint16_t tim;
uint8_t x;
if(x=bit_is_set(PINE,PE4))
	{
	printP(PSTR("released\n"));
	tim=timer0x;
	tim = (tim<<8)|TCNT0;
	srand(rand()^tim);
	}
else
	{
	srand(rand()^TCNT0);
	TCNT0=0;
	timer0x=0;
	}
if(!x) return;
if(tim>20000) //actually this timing should depend on XTAL
	{
	eeprom_write_word(&counter_eeprom_output,counter_output);
	//instead, alarmserver stores counters in a file
	//uint8_t i; for(i=0; i<NACTORS; ++i) eeprom_write_dword(counters_stored+i,0);
	eeprom_write_word(&ee_tempmin,1000);
	eeprom_write_word(&ee_tempmax, -1000);
	//counters for external termometers are in RAM and are reset automatically 
	watchdog(1); while(1){}; //reset
	}
process_press(tim>5000);
}

INTERRUPT(SIG_OVERFLOW0)
{
++timer0x;
}

static volatile uint16_t counter2 =0;

INTERRUPT(SIG_OVERFLOW2)
{
if(counter2) --counter2; //frequency XTAL/1024/256
}





init_rotcoder(void)
{
//start timer0
TCNT0=0;
TCCR0=CK1024;
OCR0=0;
sbi(TIMSK,TOIE0);

//pull-up input pins
cbi(DDRE,PE4);
cbi(DDRE,PE6);
cbi(DDRE,PE7);
sbi(PORTE,PE4);
sbi(PORTE,PE6);
sbi(PORTE,PE7);

//and allow any-edge interrupts on INT4,6,7
sbi(EICRB,ISC40); cbi(EICRB,ISC41);
sbi(EICRB,ISC60); cbi(EICRB,ISC61);
sbi(EICRB,ISC70); cbi(EICRB,ISC71);
sbi(EIMSK,INT4);
sbi(EIMSK,INT6);
sbi(EIMSK,INT7);
}


//delay routines
void delay_xs(uint16_t xs) __attribute__((noinline));

void delay_xs(uint16_t xs) //xs takes 4 clocks time
{
        asm volatile (
        "\n"
        "L_dl1%=:" "\n\t"
        "sbiw %0, 1" "\n\t"
        "brne L_dl1%=" "\n\t"
        : "=&w" (xs)
        );
        return;
}

void delay_ms(uint16_t ms) __attribute__((noinline));

void delay_ms(uint16_t ms) //ms takes 1/1000 s
{
while(ms--) delay_xs(XTAL/4/1000);
}


//display routines

#define lcd_delay 50

void lcd_w4bit(uint8_t rs, uint8_t x)
{
PORTC= 1<<PC4|light
	 | rs <<PC6;
delay_xs(lcd_delay);
PORTC |= x&0x0f;
delay_xs(lcd_delay);
cbi(PORTC,PC4);
delay_xs(lcd_delay);
sbi(PORTC,PC4);
delay_xs(lcd_delay);
}

uint8_t lcd_r4bit(uint8_t rs)
{
uint8_t r;
PORTC= 1<<PC4 | 1<<PC5|light
         | rs <<PC6;
delay_xs(lcd_delay);
cbi(PORTC,PC4);
delay_xs(lcd_delay);
r=PINC&0x0f;
cbi(PORTC,PC5);
delay_xs(lcd_delay);
return r;
}

uint8_t lcd_rbyte(uint8_t rs)
{
DDRC=0xf0;
delay_xs(lcd_delay);
return (lcd_r4bit(rs)<<4)|lcd_r4bit(rs);
DDRC=0xff;
delay_xs(lcd_delay);
}

void lcd_init4bit(void)
{
DDRC=0xff;
delay_xs(20000);
lcd_w4bit(0,3);
delay_xs(10000);
lcd_w4bit(0,3);
delay_xs(500);
lcd_w4bit(0,3);
lcd_w4bit(0,2);
}

void lcd_wbyte(uint8_t rs, uint8_t x)
{
lcd_w4bit(rs,x>>4);
lcd_w4bit(rs,x);
}

void lcd_clear(void)
{
lcd_wbyte(0,0x01);
delay_xs(30000);
}

void lcd_init(void)
{
lcd_init4bit();
lcd_wbyte(0,0x28);
lcd_wbyte(0,0x08);
lcd_clear();
lcd_wbyte(0,0x06);
lcd_wbyte(0,0x0c);
}

void lcd_print(char *t)
{
while(*t) if(*t=='\n') {++t; lcd_wbyte(0,0xc0);} else lcd_wbyte(1,*t++);
}


void lcd_print8(char *t)
{
uint8_t l=0;
while(*t) {lcd_wbyte(1,*t++); ++l;}
while(l<8) {lcd_wbyte(1,' '); ++l;}
}

void lcd_cursor(uint8_t r, uint8_t c)
{
lcd_wbyte(0,0x80+c+(r<<6));
}


//DS18B20 alternative temperature routines 
//adapted from the code of Davide Gironi, using internal pullup
/*
ds18b20 lib 0x02
copyright (c) Davide Gironi, 2012
Released under GPLv3.
Please refer to LICENSE file for licensing information.

References:
  + Using DS18B20 digital temperature sensor on AVR microcontrollers
    by Gerard Marull Paretas, 2007
    http://teslabs.com/openplayer/docs/docs/other/ds18b20_pre1.pdf
*/
//connection
#define DS18B20_PORT TEMPPORTOUT
#define DS18B20_DDR TEMPDDR
#define DS18B20_PIN TEMPPORT
#define DS18B20_DQ TEMPPIN
//commands
#define DS18B20_CMD_CONVERTTEMP 0x44
#define DS18B20_CMD_RSCRATCHPAD 0xbe
#define DS18B20_CMD_WSCRATCHPAD 0x4e
#define DS18B20_CMD_CPYSCRATCHPAD 0x48
#define DS18B20_CMD_RECEEPROM 0xb8
#define DS18B20_CMD_RPWRSUPPLY 0xb4
#define DS18B20_CMD_SEARCHROM 0xf0
#define DS18B20_CMD_READROM 0x33
#define DS18B20_CMD_MATCHROM 0x55
#define DS18B20_CMD_SKIPROM 0xcc
#define DS18B20_CMD_ALARMSEARCH 0xec

#define _delay_us(us) delay_xs((XTAL/1000)*(us)/4/1000);
//if more accuracy is needed, use other routine with inline __attribute__((gnu_inline))


/*
 * ds18b20 init
 */
uint8_t ds18b20_reset() {
	uint8_t i;

	//low for 480us
	DS18B20_PORT &= ~ (1<<DS18B20_DQ); //low
	DS18B20_DDR |= (1<<DS18B20_DQ); //output
	_delay_us(480);

	//release line and wait for 60uS
	DS18B20_DDR &= ~(1<<DS18B20_DQ); //input
	DS18B20_PORT |= (1<<DS18B20_DQ); //pullup
	_delay_us(60);

	//get value and wait 420us
	i = (DS18B20_PIN & (1<<DS18B20_DQ));
	_delay_us(420);

	//return the read value, 0=ok, 1=error
	return i;
}

/*
 * write one bit
 */
void ds18b20_writebit(uint8_t bit){
	//low for 1uS
	DS18B20_PORT &= ~ (1<<DS18B20_DQ); //low
	DS18B20_DDR |= (1<<DS18B20_DQ); //output
	_delay_us(1);

	//if we want to write 1, release the line (if not will keep low)
	if(bit)
		{
		DS18B20_DDR &= ~(1<<DS18B20_DQ); //input
		DS18B20_PORT |= (1<<DS18B20_DQ); //pullup
		}

	//wait 60uS and release the line
	_delay_us(60);
	DS18B20_DDR &= ~(1<<DS18B20_DQ); //input
	DS18B20_PORT |= (1<<DS18B20_DQ); //pullup
}

/*
 * read one bit
 */
uint8_t ds18b20_readbit(void){
	uint8_t bit=0;

	//low for 1uS
	DS18B20_PORT &= ~ (1<<DS18B20_DQ); //low
	DS18B20_DDR |= (1<<DS18B20_DQ); //output
	_delay_us(1);

	//release line and wait for 14uS
	DS18B20_DDR &= ~(1<<DS18B20_DQ); //input
	DS18B20_PORT |= (1<<DS18B20_DQ); //pullup
	_delay_us(14);

	//read the value
	if(DS18B20_PIN & (1<<DS18B20_DQ))
		bit=1;

	//wait 45uS and return read value
	_delay_us(45);
	return bit;
}

/*
 * write one byte
 */
void ds18b20_writebyte(uint8_t byte){
	uint8_t i=8;
	while(i--){
		ds18b20_writebit(byte&1);
		byte >>= 1;
	}
}

/*
 * read one byte
 */
uint8_t ds18b20_readbyte(void){
	uint8_t i=8, n=0;
	while(i--){
		n >>= 1;
		n |= (ds18b20_readbit()<<7);
	}
	return n;
}

uint8_t ds_crc8(uint8_t crc, uint8_t *data, uint16_t len)
{
while(len--) 
	{
               uint8_t inbyte = *data++;
	 	uint8_t i;
               for (i = 8; i; i--) {
                        uint8_t mix = (crc ^ inbyte) & 0x01;
                        crc >>= 1;
                        if (mix) crc ^= 0x8C;
                        inbyte >>= 1;
                }
        }

return crc;
}


/*
 * get temperature in 1/16 C (12bit mode)
 */
int16_t ds18b20_gettemp() {
	uint8_t data[9];
	uint8_t i;
	int16_t retd = 0;

	cli();
	uint8_t thermo_present = ! ds18b20_reset(); //reset
	if(!thermo_present)  printP(PSTR("no thermometer found\n"));
	if(!thermo_present) return UNDEFINED_TEMP;
	ds18b20_writebyte(DS18B20_CMD_SKIPROM); //skip ROM
	ds18b20_writebyte(DS18B20_CMD_CONVERTTEMP); //start temperature conversion
	//avoid forbidden interrupts for a long time
	for(i=0; i<6; ++i)
		{
		sei();
		delay_ms(100);
		cli();
		if(ds18b20_readbit()) goto conversion_done;
		}
	while(!ds18b20_readbit()); //wait until conversion is complete

	conversion_done:
	cli();
	ds18b20_reset(); //reset
	ds18b20_writebyte(DS18B20_CMD_SKIPROM); //skip ROM
	ds18b20_writebyte(DS18B20_CMD_RSCRATCHPAD); //read scratchpad
	//read scratchpad including crc
	for(i=0; i<=8; ++i) data[i] = ds18b20_readbyte();
	sei();

	uint8_t crc=ds_crc8(0,data,8);
	if(crc!=data[8])
		{
		char z[16];
		itoa16(crc,z);
		printP(PSTR("crc our ")); print(z); printP(PSTR("\n"));
		itoa16(data[8],z);
        	printP(PSTR("crc DS ")); print(z); printP(PSTR("\n"));
		return UNDEFINED_TEMP;
		}

	//convert the 12 bit value obtained
	retd = data[1];
	retd <<= 8;
	retd |= data[0];
	return retd;
}

//DS18B20 todo: explicitly configure for the 12bit mode, select from more sensors



//SMT temperature routines
static uint8_t have_smt160;
static uint8_t have_ds18b20;

uint8_t check_smt160()
{
watchdog(1);
cbi(TEMPDDR,TEMPPIN);
sbi(TEMPPORTOUT,TEMPPIN); //pullup to prevent random noise
uint8_t last,l;
last= bit_is_set(TEMPPORT,TEMPPIN);
uint16_t i,pulses=0;
for(i=0; i<1000; ++i)
        {
        delay_xs(XTAL/8000);
        if((l=bit_is_set(TEMPPORT,TEMPPIN))^last) ++pulses;
        last=l;
        }
cbi(TEMPPORTOUT,TEMPPIN);
return pulses>100?1:0;
}


#ifdef EXTRA_AVERAGE
int16_t local_temperature0(void) // thermometer returns in 0.1 C
#else
int16_t local_temperature(void) // thermometer returns in 0.1 C
#endif
{
if(have_ds18b20) //assume DS18B20
{
int16_t t;
t= ds18b20_gettemp();
if(t==UNDEFINED_TEMP) return t;
t*= 5;
t/= 8;
return t;
}
if(have_smt160) //SMT160
{
int16_t t;
uint8_t i;
uint16_t lcount,hcount;

watchdog(1);

cbi(SREG,7); //switch off interrupts

//SMT160-30 duty=0.320+0.00470*T, T=(duty-0.32)/0.00470


lcount=hcount=0;


{
uint8_t i;
for(i=
#if (XTAL == 3686400L)
0
#else
#if (XTAL == 14745600L)
64
#else
128
#endif
#endif
; --i;)
{

//optimized compilation results in irreproducible and wrong timings, therefore inline assembly

asm volatile (

//loop_until_bit_is_set(TEMPPORT,TEMPPIN);
"TEMP1:\n\t"
"sbis %4,%5" "\n\t"
"rjmp TEMP1" "\n\t"

//loop_until_bit_is_clear(TEMPPORT,TEMPPIN);
"TEMP2:\n\t"
"sbic %4,%5" "\n\t"
"rjmp TEMP2" "\n\t"

//while(bit_is_clear(TEMPPORT,TEMPPIN)) ++lcount;
"TEMP3:\n\t"
"adiw %1,1" "\n\t"
"sbis %4,%5" "\n\t"
"rjmp TEMP3" "\n\t"

//while(bit_is_set(TEMPPORT,TEMPPIN)) ++hcount;
"TEMP4:\n\t"
"adiw %0,1" "\n\t"
"sbic %4,%5" "\n\t"
"rjmp TEMP4" "\n\t"

//parameter lists
: "=w" (hcount), "=w" (lcount) //outputs
: "0" (hcount), "1" (lcount), "M" ((uint8_t) &(TEMPPORT) - 0x20), "I" (TEMPPIN) //inputs
: "cc"  //clobber
);

}
}

#ifdef debug2
{
uint32_t x;
x=10000; x*=hcount;
x /=((uint32_t)hcount+lcount);
char c[8];
itoa10(x,c);
printP(PSTR("10000*Duty=")); print(c); printP(PSTR("\n"));
}
#endif

sbi(SREG,7);

{
uint32_t x;
x=68085L; x*=hcount;
x /=((uint32_t)hcount+lcount);
t = x - 21787;
t>>= 5;
}
return t;
}
//end SMT temperature

//no sensor
return UNDEFINED_TEMP;
}
//end temperature

#ifdef EXTRA_AVERAGE
int16_t local_temperature(void) // thermometer returns in 0.1 C
{
uint8_t i;
int16_t t=0;
uint8_t nrep;
nrep= have_smt160? 16:2;
for(i=nrep; i!=0; --i) {int16_t t0=local_temperature0(); if(t0==UNDEFINED_TEMP) return t0; t+=t0; delay_ms(3);}
return t/nrep;
}
#endif

static volatile int16_t temperatures[NTHERMOMETERS];
static volatile int16_t pressures[NTHERMOMETERS];
static volatile int16_t lights[NTHERMOMETERS];
static volatile int8_t humidities[NTHERMOMETERS];
static volatile uint16_t temp_timeouts[NTHERMOMETERS];
static volatile uint16_t azimuths[NTHERMOMETERS];
static volatile uint16_t windspeeds[NTHERMOMETERS];
static volatile uint16_t rains[NTHERMOMETERS];
#define TIMEOUTS_THR 20000
static uint16_t temp_timeouts_thr[NTHERMOMETERS] __attribute__ ((section(".eeprom"))) = {20000,20000,20000,20000,20000,20000,20000,20000,20000};
static uint8_t ee_eff_temp_from  __attribute__ ((section(".eeprom"))) = SENSOR_SERIAL; //local one
static uint8_t ee_eff_temp_to __attribute__ ((section(".eeprom"))) = SENSOR_SERIAL; //local one
static uint8_t ee_eff_temp_kind __attribute__ ((section(".eeprom"))) = AVERAGE;


int16_t effective_temperature(void)
{
int16_t temp;
uint8_t i,c;
switch(eff_temp_kind) {
case AVERAGE:
	c=0;
	temp=0;
	for(i=eff_temp_from; i<=eff_temp_to; ++i)
		{
		if(temperatures[i]!=UNDEFINED_TEMP)
			{
			++c;
			temp += temperatures[i];
			}
		else return UNDEFINED_TEMP;
		}
	return c==0?UNDEFINED_TEMP:temp/c;
	break;
case MINIMUM:
	temp= -UNDEFINED_TEMP;
	for(i=eff_temp_from; i<=eff_temp_to; ++i)
		if(temperatures[i]!=UNDEFINED_TEMP)
			{
			if(temperatures[i]<temp) temp=temperatures[i];
			}
		else return UNDEFINED_TEMP;
	if(temp== -UNDEFINED_TEMP) return UNDEFINED_TEMP;
	return temp;
	break;
case MAXIMUM:
	temp=UNDEFINED_TEMP;
	for(i=eff_temp_from; i<=eff_temp_to; ++i)
	    if(temperatures[i]!=UNDEFINED_TEMP)
		{
                if(temperatures[i]>temp) temp=temperatures[i];
		}
	    else return UNDEFINED_TEMP;
	return temp;
	break;
default:
	return local_temperature();
	break;
	}
}


void init_random_byte(void)
{
//start timer0
TCNT0=local_temperature();
TCCR0=1;

}

uint8_t random_byte(void)
{
//read value of timer0
return rand()^TCNT0;
}

void txcw(void)
{
sbi(DDRB,PB7); sbi(PORTB,PB7);
sbi(PORTB,PB5); sbi(DDRB,PB5);
uint8_t i;
for(i=0; i<100; ++i) {watchdog(1); delay_ms(100);}

cbi(DDRB,PB7); cbi(PORTB,PB7);
cbi(PORTB,PB5); cbi(DDRB,PB5);
}

void do_transmit(uint8_t onoff)
{
if(onoff)
        {
		sbi(DDRB,PB7); sbi(PORTB,PB7); 
                cbi(PORTB,PB5); sbi(DDRB,PB5);
	delay_ms(1);
	TCCR1A= (1<<WGM11);
 	TCCR1B= (1<<WGM12)|(1<<WGM13)|CK;
	ICR1=(uint16_t)(XTAL*12./10000+.5)-1;
	OCR1A=0;
	TCNT1=0;
	sbi(TIMSK,TOIE1);
	}
else
	{
	TCCR1A=0;
        TCCR1B=0;
        cbi(TIMSK,TOIE1);
        cbi(DDRB,PB7); cbi(PORTB,PB7);
                cbi(PORTB,PB5); cbi(DDRB,PB5);
	}
}




static uint8_t receiving=0;
void do_receive(uint8_t onoff)
{
receiving=onoff;
if(onoff)
	{
	TCCR1B=CK; //no prescale
	TCCR1A=0;
	OCR1B=XTAL/10000/oversampling; //minimum recommended sampling = 10000
	//activate interrupts for receive
	TCNT1=0;
	sbi(TIMSK,OCIE1B);
	}
else
	{
	cbi(TIMSK,OCIE1B);
	TCCR1B=0; //no prescale
        TCCR1A=0;
        OCR1B=0;
	}
}


//statics for output keeloq processing
static KEELOQ_CODE2 keeloq_crypt_output= LOCATION_KEELOQ_SENSOR(SENSOR_SERIAL);

//and for input keeloq processing
static KEELOQ_CODE2 keeloq_actor_crypts[] = LOCATION_KEELOQ_ACTORS_ALL;
static KEELOQ_CODE2 keeloq_sensor_crypts[] = LOCATION_KEELOQ_SENSORS_ALL;
KEELOQ_CODE2 *keeloq_all[MAX_DEVICE_TYPES] = {NULL,keeloq_sensor_crypts,keeloq_actor_crypts};
static uint32_t actors_resync[NACTORS]; //zero initialized
static uint32_t actors_counters[NACTORS]; //zero initialized
static uint32_t sensors_resync[NSENSORS]; //zero initialized at reset, no eeprom backup
static uint32_t sensors_counters[NSENSORS]; //eeprom backup once in a while to avoid excessive wear
static uint32_t sensors_counters_init[NSENSORS]  __attribute__ ((section(".eeprom"))) = {0,COUNTER_SENSOR1_INIT,COUNTER_SENSOR2_INIT,COUNTER_SENSOR3_INIT};

#define EEPROM_MAGIC 12345
static uint16_t eeprom_magic  __attribute__ ((section(".eeprom"))) = EEPROM_MAGIC;

void set_factory_defaults(void)
{
eeprom_write_word(&ee_tempset,TEMPSET_DEFAULT);
eeprom_write_byte(&ee_hysteresis_on,HYSTERESIS_ON_DEFAULT);
eeprom_write_byte(&ee_hysteresis_off,HYSTERESIS_OFF_DEFAULT);
eeprom_write_word(&ee_tempmin,TEMPMIN_INIT);
eeprom_write_word(&ee_tempmax,TEMPMAX_INIT);
eeprom_write_word(&ee_nreset,0);
eeprom_write_byte(&ee_activestate,ACTIVESTATE_DEFAULT);
uint8_t i;
for(i=0; i<NTHERMOMETERS; ++i) eeprom_write_word(&temp_timeouts_thr[i],TIMEOUTS_THR);
eeprom_write_byte(&ee_eff_temp_from,SENSOR_SERIAL);
eeprom_write_byte(&ee_eff_temp_to,SENSOR_SERIAL);
eeprom_write_byte(&ee_eff_temp_kind,AVERAGE);

eeprom_write_word(&eeprom_magic,EEPROM_MAGIC);
printP(PSTR("Reset to factory defaults has been done.\n"));
}


#define min(x,y) ((x)<(y)?(x):(y))


//returns -1 or the serial number (which should be compared to ACTOR_SERIAL to determine if it is for us)
//actually the serial could be checked first, but rewrite of the receive_message etc. would be necessary...

int8_t receive(CODE433 *cc, DEVICE *device)
{
message msg;
uint8_t  r;
if(!incodelen) return -1;
#ifdef debug
{
uint8_t len,i;
        char c[8];
len=incodelen;
 itoa10(len,c); print(c); print(": ");
        for(i=0; i<(len+7)>>3; ++i)
                {
                itoa16(incode[i],c);
                print(c);
                print(" ");
                }
        print("\n");
}
#endif

if(incodelen2==incodelen)
                {
                //same message received twice
		if(!memcmp(incode2,incode,min(MESSAGE_BYTES+1,(incodelen2+7)/8))) {incodelen=0; return -1;}
                }
printP(PSTR("unique msg received\n"));
incodelen2=incodelen;
memcpy(incode2,incode,min(MESSAGE_BYTES+1,(incodelen2+7)/8));
watchdog(1);
r=decrypt_message(&msg,incode,incodelen,keeloq_all);
incodelen=0; //allow receive to the buffer again
if(r) return -1;
printP(PSTR("decrypt_message OK\n"));

SERIAL serial;
uint32_t counter;
watchdog(1);
r=disassemble_message32(device,&serial,&counter,&cc->code,&msg);
if(r) return -1;
printP(PSTR("disassemble_message OK\n"));

switch(*device) {

case SENSOR_DEVICE:
#ifdef debug
        {
        char c[16];
	itoa10(serial,c);
	printP(PSTR("sensor serial ")); print(c);
        itoa10(counter,c);
        printP(PSTR(" sensor counter "));print(c);
        itoa10(sensors_counters[serial],c);
        printP(PSTR(" sensor counters ")); print(c); print("\n");
        }
#endif
	if(serial==SENSOR_SERIAL) return -1; //somebody is transmitting under our ID!
	r=validate_message32(SENSOR_DEVICE, NSENSORS, sensors_counters, sensors_resync, *device, serial, counter,cc->cmd.instr &LONGCOUNTER);
	if(r)
                {
                char c[6];
                itoa10(r,c);
                printP(PSTR("Validate error ")); print(c); print("\n");
                return -1;
                }
	printP(PSTR("validated OK\n"));
        return serial;
	break;

case ACTOR_DEVICE:
	for(r=0; r<NACTORS; ++r) actors_counters[r]=eeprom_read_dword(counters_stored+r);
#ifdef debug
	{
	char c[16];
	itoa10(serial,c);
	printP(PSTR("actor serial "));print(c);
	itoa10(counter,c);
	printP(PSTR(" counter "));print(c); 
	itoa10(actors_counters[serial],c);
	printP(PSTR(" counters ")); print(c); print("\n");
	}
#endif
	r=validate_message32(ACTOR_DEVICE, NACTORS, actors_counters, actors_resync, *device, serial, counter,cc->cmd.instr &LONGCOUNTER);
	if(r)
                {
                char c[6];
                itoa10(r,c);
                printP(PSTR("Validate error ")); print(c); print("\n");
		return -1;
                }
	eeprom_write_dword(counters_stored+serial,actors_counters[serial]);
	printP(PSTR("validated OK\n"));
	return serial;
case KEYBOARD_DEVICE:
	printP(PSTR("keyboards not relevant\n"));
	return -1;
default:
	printP(PSTR("unknown device type\n"));
	return -1;
}

}



void transmit(uint32_t code, uint8_t count)
{

if(!count) return;

//wait for end of previous transmit or switch on transmitter
if(!receiving  && (txcount || outcodelen))
        {
		do {watchdog(1);delay_ms(10);} while(txcount || outcodelen);
        }
else
	{
	//switch on transmitter
	do_receive(0);
	do_transmit(1);
	}


#ifdef debug
{
printP(PSTR("before increment counter = \n"));
char c[8];
itoa10(counter_output,c); print(c);
printP(PSTR(" eeprom= "));
itoa10(eeprom_read_word(&counter_eeprom_output),c); print(c);
printP(PSTR("\n"));
}
#endif
++counter_output;
if(counter_output<=eeprom_read_word(&counter_eeprom_output)) 
	{
	counter_output= eeprom_read_word(&counter_eeprom_output)+256; //if bug, skip to next from eeprom +offset to recognize what happened
	}
if(counter_output > eeprom_read_word(&counter_eeprom_output) + 200) eeprom_write_word(&counter_eeprom_output,counter_output);

#ifdef debug
{
printP(PSTR("before assemble message = \n"));
char c[8];
itoa10(counter_output,c); print(c);
printP(PSTR(" eeprom= "));
itoa10(eeprom_read_word(&counter_eeprom_output),c); print(c);
printP(PSTR("\n"));
}
#endif


//assemble message
message msg;
assemble_message(SENSOR_DEVICE,SENSOR_SERIAL,counter_output,code,random_byte(),&msg);

#ifdef debug
{
printP(PSTR("after assemble message = \n"));
char c[8];
itoa10(counter_output,c); print(c);
printP(PSTR(" eeprom= "));
itoa10(eeprom_read_word(&counter_eeprom_output),c); print(c);
printP(PSTR("\n"));
}
#endif


watchdog(1);

//encrypt message, result in outcode
encrypt_message(&msg, outcode, keeloq_crypt_output);


//transmit the code a few times
txcount=count;
outcodelen=MESSAGE_BYTES*8;
printP(PSTR("Transmitting\n"));

//do not wait for end, loop in main checks and starts receiving when finished
}


void transmit_actor(uint8_t serial, uint32_t code, uint8_t count)
{

if(!count) return;

//wait for end of previous transmit or switch on transmitter
if(!receiving  && (txcount || outcodelen))
        {
		do {watchdog(1);delay_ms(10);} while(txcount || outcodelen);
        }
else
	{
	//switch on transmitter
	do_receive(0);
	do_transmit(1);
	}


//increment counter
++actors_counters[serial];
eeprom_write_word(counters_stored+serial,actors_counters[serial]);

//assemble message
message msg;
assemble_message(ACTOR_DEVICE,serial,actors_counters[serial],code,random_byte(),&msg);



watchdog(1);

//encrypt message, result in outcode
encrypt_message(&msg, outcode, keeloq_actor_crypts[serial]);


//transmit the code a few times
txcount=count;
outcodelen=MESSAGE_BYTES*8;
printP(PSTR("Transmitting\n"));

//do not wait for end, loop in main checks and starts receiving when finished
}


void printpres(int16_t pres, char *t)
{
if(pres==UNDEFINED_PRESSURE)
        {
        strcpy(t,"N/A");
        return;
        }
t[5]= '0' + pres%10; pres/=10;
t[4]='.';
t[3]= '0' + pres%10; pres/=10;
t[2]= '0' + pres%10; pres/=10;
t[1]= '0' + pres%10; pres/=10;
if(pres) t[0]= '0' + pres; else t[0]=' ';
}

void printhum(int8_t hum, char *t)
{
if(hum==UNDEFINED_HUMID)
        {
        strcpy(t,"N/A");
        return;
        }
t[3]='%';
t[2]='0' + hum%10; hum/=10;
if(hum) {t[1]='0' + hum%10; hum/=10;} else t[1]=' ';
if(hum) t[0]='0' + hum%10;  else t[0]=' ';
}


void printtemp(int16_t temp, char *t)
{
if(temp==UNDEFINED_TEMP)
	{
	strcpy(t,"N/A");
	return;
	}
uint8_t sign = (temp<0);
if(sign) temp= -temp;
strcpy(t,"   . C");
t[4] = '0' + temp%10;
temp /= 10;
t[2] = '0' + temp%10;
temp /= 10;
if(temp)
	{
	t[1] = '0' + temp%10;
	if(sign) t[0]='-';
	}
else
	{
	if(sign) t[1]='-';
	}
}


volatile uint8_t heater_on=0;
volatile uint8_t reportheatcool=0;

void heater(uint8_t onoff)
{
#ifdef debug
printP(PSTR("in heater\n"));
#endif
if(onoff) {sbi(PORTA,PA0); sbi(PORTA,PA3); sbi(PORTA,PA4); reportheatcool|=HEATER_ON;}
else	  {cbi(PORTA,PA0); cbi(PORTA,PA3); cbi(PORTA,PA4); reportheatcool|=HEATER_OFF;}
heater_on=onoff;
}

volatile uint8_t cooler_on=0;

void cooler(uint8_t onoff)
{
#ifdef debug
printP(PSTR("in cooler\n"));
#endif

if(onoff) {sbi(PORTA,PA1); sbi(PORTA,PA5); reportheatcool|=COOLER_ON;}
else      {cbi(PORTA,PA1); cbi(PORTA,PA5); reportheatcool|=COOLER_OFF;}
cooler_on=onoff;
}


static int16_t reported_temp;
void send_temp_report(uint8_t arg)
{
#ifdef debug2
{
char z[16];
printP(PSTR("temp[] "));
itoa10(temperatures[SENSOR_SERIAL],z);
print(z);
printP(PSTR(" reported "));
itoa10(reported_temp,z);
print(z);
printP(PSTR("\n"));
}
#endif
CODE433 c;
c.cmd.instr= THERMOMETER|TEMPERATURE_REPORT;
c.cmd.arg1=arg;
reported_temp=temperatures[SENSOR_SERIAL]; //not the effective temerature
c.cmd.arg2=reported_temp;
printP(PSTR("send_temp_report\n"));
transmit(c.code,TXCOUNT);
}

void reset_heater(void)
{
                cbi(PORTA,PA2);
                {uint8_t i; for(i=0; i<30; ++i) {watchdog(1); delay_ms(100);}}
                sbi(PORTA,PA2);
}




int main(void)
{
UART_INIT(BAUD); printP(PSTR("Reset!\n"));

reported_temp=UNDEFINED_TEMP;

{uint16_t test = eeprom_read_word(&eeprom_magic);
if(test!=EEPROM_MAGIC) set_factory_defaults(); //eeprom was not programmed or became corrupted
}

{
uint16_t n = eeprom_read_word(&ee_nreset);
eeprom_write_word(&ee_nreset,n+1);
}


//allow uart interrupts
{char dummy=UDR1;}
sbi(UCSR1B,RXCIE1);



lock_tempset=0;
uint8_t report1a=0;

//reset rain counters
{uint8_t i;
for(i=0; i<NTHERMOMETERS; ++i) rains[i]=0;
}


cbi(PORTC,PC7); sbi(DDRC,PC7); light=0; //switch off light

//init temppin
cbi(TEMPDDR,TEMPPIN);
sbi(TEMPPORTOUT,TEMPPIN);

have_ds18b20 = 0;
have_smt160 = check_smt160();
if(have_smt160) printP(PSTR("SMT160 found\n"));
else printP(PSTR("SMT160 NOT found\n"));
if(!have_smt160)
	{
	have_ds18b20 = ! ds18b20_reset();
	if(have_ds18b20) printP(PSTR("DS18B20 found\n"));
	else printP(PSTR("DS18B20 NOT found\n"));
	}

uint16_t remotecounter=0;
uint8_t inquire_status=0;
watchdog(1);
tempset=eeprom_read_word(&ee_tempset);
hysteresis_on=eeprom_read_byte(&ee_hysteresis_on);
hysteresis_off=eeprom_read_byte(&ee_hysteresis_off);
activestate=eeprom_read_byte(&ee_activestate);

eff_temp_from=eeprom_read_byte(&ee_eff_temp_from);
eff_temp_to=eeprom_read_byte(&ee_eff_temp_to);
eff_temp_kind=eeprom_read_byte(&ee_eff_temp_kind);

{uint8_t i;
for(i=0; i<NSENSORS; ++i) sensors_counters[i] = eeprom_read_dword(&sensors_counters_init[i]);
}


init_random_byte();



//global interrupt activate
sbi(SREG,7);

//write initial message to the display and blink leds
lcd_init();
lcd_print("Reset!");
watchdog(1);

//activate timer2
counter2=0;
TCNT2=0;
TCCR2=5; //CK1024
sbi(TIMSK,TOIE2); //overflow interrupts



//this must be before first transmit()
counter_output=eeprom_read_word(&counter_eeprom_output)+256;
#ifdef debug
{
printP(PSTR("init counter_output to \n"));
char c[8];
itoa10(counter_output,c); print(c);
printP(PSTR("\n"));
}
#endif

do_transmit(1); delay_ms(50); //should not be necessary but be sure the TX module is powered and stabilized
//transmit reset report
        {
        CODE433 c;
        c.cmd.instr= THERMOMETER|RESET_REPORT;
        transmit(c.code,TXCOUNT);
	watchdog(1); delay_ms(500); 
	watchdog(1); delay_ms(500); 
#ifdef debug
	printP(PSTR("transmit from reset\n"));
#endif
	transmit(c.code,TXCOUNT);
        }

do{
watchdog(1);
delay_ms(100);
} while(outcodelen||txcount);
watchdog(1);
delay_ms(500);

//init relays and LEDs
PORTA=0; DDRA=0xff;
cbi(PORTE,PE3); sbi(DDRE,PE3);

sbi(DDRA,PA2); sbi(PORTA,PA2); //turn on power for heater
reset_heater_pending=0;

//enable receive
do_receive(1);

//enable rotcoder
init_rotcoder();


int16_t temp,tempold;
char tempchar='=';
{
uint8_t i;
for(i=0;i<NTHERMOMETERS; ++i) temperatures[i]= UNDEFINED_TEMP;
}
int16_t tempstor[AVERAGING];
uint8_t tempindex;
temp=local_temperature();
for(tempindex=0;tempindex<AVERAGING; ++tempindex) tempstor[tempindex]=temp;
tempold=temp=UNDEFINED_TEMP;
tempindex=0;
uint16_t pass=0;

//MAIN LOOP
while(1)
{
++pass;
++remotecounter;
++clearingcounter;

//switch off timed relay
if(counter2==0) cbi(PORTE,PE3);

#ifdef debug
if((pass&15)==0)
{
char c[8];
itoa10(pass,c);
print("pass "); print(c); print("\n");
}
#endif

if(reset_heater_pending)
	{
	reset_heater_pending=0;
	reset_heater();
	}
{
uint8_t i;
for(i=0; i<NTHERMOMETERS; ++i) {if(temp_timeouts[i]!=0xffff) ++temp_timeouts[i];}
}

if(!receiving && !txcount && !outcodelen)
        {
        delay_ms(5);
        do_transmit(0);
        delay_ms(5);
        do_receive(1);
        }

delay_ms(50);
char text[10];

//check incoming commands from uart
if(uartline)
	{
	switch(inuart[0]) {
	case 'D': //factory defaults
		if(strncmp(inuart,"Defaults",8)) break; //prevent accidental erase by typo
		set_factory_defaults();
		break;
	case 'R': //reset heater
		reset_heater();
		break;
	case 'r':
		{
		printP(PSTR("Number of resets since last eeprom programming = "));
		char c[16];
		itoa10(eeprom_read_word(&ee_nreset),c); print(c);
		printP(PSTR("\n"));
		}
		break;
	case 'C':
		{
		uint16_t c = atoi(inuart+2);
                {
                CODE433 cc;
                inquire_status=0;
                cc.cmd.instr= THERMOMETER|COUNTER_OUTPUT_REPORT;
                cc.cmd.arg2 = c;
                cc.cmd.arg1= 0;
#ifdef debug
		printP(PSTR("transmit from uart command\n"));
#endif
                transmit(cc.code,TXCOUNT);
                }
		eeprom_write_word(&counter_eeprom_output,c);
		counter_output=c;
#ifdef debug
{
printP(PSTR("setting counter_output = \n"));
char c[8];
itoa10(counter_output,c); print(c);
printP(PSTR(" eeprom= "));
itoa10(eeprom_read_word(&counter_eeprom_output),c); print(c);
printP(PSTR("\n"));
}
#endif
		printP(PSTR("\nOK\n"));
		}
		break;
	case 'L':
		{
		char c[16];
		itoa10(eeprom_read_word(&counter_eeprom_output),c);
		printP(PSTR("eeprom output counter ")); print(c); print("\n");
		itoa10(counter_output,c);
        	printP(PSTR("output counter ")); print(c); print("\n");
                }
		break;
	case 'O':
		{
		uint8_t i;
		for(i=0; i<NTHERMOMETERS; ++i)
			{
			char c[16];
                	itoa10(eeprom_read_word(&temp_timeouts_thr[i]),c);
                	printP(PSTR("eeprom temp_timeouts_thr ")); print(c); print("\n");
                	itoa10(temp_timeouts[i],c);
                	printP(PSTR("temp_timeouts ")); print(c); print("\n");
			}
		}
		break;
	case 'o':
		{
                uint8_t i = inuart[1]-'0';
                uint16_t c = atoi(inuart+3);
                if(i>=NTHERMOMETERS) break;
                eeprom_write_word(&temp_timeouts_thr[i],c);
                printP(PSTR("\nOK\n"));
                }
		break;
	case 'c':
		{
		uint8_t i = inuart[1]-'0';
		uint16_t c = atoi(inuart+3);
		if(i>=NSENSORS) break;
		eeprom_write_dword(&sensors_counters_init[i],c);
		sensors_counters[i] = c;
		printP(PSTR("\nOK\n"));
		}
		break;
	case 'l':
		{
                uint8_t i = inuart[1]-'0';
		if(i>=NSENSORS) break;
        char c[16];
        itoa10(i,c);
        printP(PSTR("sensor serial ")); print(c);
        itoa10(eeprom_read_dword(&sensors_counters_init[i]),c);
        printP(PSTR(" sensor counter init "));print(c);
        itoa10(sensors_counters[i],c);
        printP(PSTR(" sensor counter ")); print(c); print("\n");
		}
		break;
	case 'a':
		{
                uint8_t i = inuart[1]-'0';
		if(i>=NACTORS) break;
        char c[16];
        itoa10(i,c);
        printP(PSTR("actor serial ")); print(c);
        itoa10(eeprom_read_dword(&counters_stored[i]),c);
        printP(PSTR(" actor counter in eeprom "));print(c);
	itoa10(actors_counters[i],c);
	printP(PSTR(" and in ram "));print(c);
		}
		break;
	case 't':
		{
		int16_t t=effective_temperature();
		printtemp(t, text);
                printP(PSTR("T= ")); print(text); print("\n");		
		}
		break;
	case 'T':
		send_temp_report(0);
		do{watchdog(1);delay_ms(100);} while(txcount!=0 || outcodelen!=0); 
		printP(PSTR("Temperature report sent.\n"));
		break;
	case 'w':
		printP(PSTR("TXCW\n"));
		txcw();
		break;
	default:
		printP(PSTR("\nUnknown command\n"));
	}
	uartline=0;
	}

watchdog(1);
if((pass& MEASUREMODULO)==0)
	{
	if(txcount==0 && outcodelen==0 ) //check temperature every 7-th step, but not when transmitting
		{
		tempstor[tempindex]=local_temperature(); //check temperature not each step and average anyway
		tempindex = (tempindex+1)%AVERAGING;
		int16_t tmp=0;
		{uint8_t i;
		for(i=0; i<AVERAGING; ++i) tmp +=tempstor[i];
		}
		if(SENSOR_SERIAL>=0 && SENSOR_SERIAL < NTHERMOMETERS)
			{
			temperatures[SENSOR_SERIAL]=tmp/AVERAGING;
			temp_timeouts[SENSOR_SERIAL]=0;
			}
		}

	temp=effective_temperature();
	tempchar='=';
	watchdog(1);
	if(temp==UNDEFINED_TEMP)
		{
		temp=local_temperature(); //backup if remote sensors fail
		tempchar='!';
		}
	if(temp!=tempold)
		{
		char tempchar2[2];
		tempchar2[0]=tempchar;
		tempchar2[1]=0;
		printtemp(temp, text);
		print("T"); print(tempchar2); print(" "); print(text); print("\n");
		tempold=temp;
		}

	if(temp> (int16_t) eeprom_read_word(&ee_tempmax)) eeprom_write_word(&ee_tempmax,temp);
	if(temp< (int16_t) eeprom_read_word(&ee_tempmin)) eeprom_write_word(&ee_tempmin,temp);
	if(++tempcount == 0) tempsum=0; else tempsum += temp;
	}

//control display according state set by asynchronous events
watchdog(1);
switch (menustate) {
	case 8: 
		lcd_cursor(0,0);
                lcd_print8(menulabel[menustate-2]);
		lcd_cursor(0,4);
		lcd_print("    ");
		lcd_cursor(0,5);
		itoa10(tempselect,text);
		lcd_print(text);
		lcd_cursor(1,0);
		text[0]='T';
                text[1]=tempchar;
                printtemp(temperatures[tempselect],text+2);
                text[8]=0;
                lcd_print8(text);
		break;
	case 9:
		lcd_cursor(0,0);
                lcd_print8(menulabel[menustate-2]);
                lcd_cursor(1,0);
                strcpy(text,"FROM=");
                itoa10(eff_temp_from,text+5);
		lcd_print8(text);
                break;
	case 10:
		lcd_cursor(0,0);
                lcd_print8(menulabel[menustate-2]);
                lcd_cursor(1,0);
                strcpy(text,"TO=");
                itoa10(eff_temp_to,text+3);
                lcd_print8(text);
                break;
	case 11:
		lcd_cursor(0,0);
                lcd_print8(menulabel[menustate-2]);
                lcd_cursor(1,0);
                lcd_print8(eff_temp_label[eff_temp_kind]);
		break;
	case 12: //reset heater
		lcd_cursor(0,0);
                lcd_print8(menulabel[menustate-2]);
                lcd_cursor(1,0);
                lcd_print8("push now");
		break;
	case 13: //show no. of resets
		lcd_cursor(0,0);
                lcd_print8(menulabel[menustate-2]);
                lcd_cursor(1,0);
		itoa10(eeprom_read_word(&ee_nreset),text);
                lcd_print8(text);
		break;
	case 15: //lock
		lcd_cursor(0,0);
                lcd_print8(menulabel[menustate-2]);
		lcd_cursor(1,0);
		lcd_print8(lock_tempset?"LOCKED":"UNLOCKED");
		break;
        case 16: //heatwater
                lcd_cursor(0,0);
                lcd_print8(menulabel[menustate-2]);
                lcd_cursor(1,0);
                if(heatwater==0) strcpy(text,"T=AUTO");
                else {text[0]='T'; text[1]='='; printtemp(10*heatwater,text+2);}
                lcd_print8(text);
                break;
        case 17: //tapwater
                lcd_cursor(0,0);
                lcd_print8(menulabel[menustate-2]);
                lcd_cursor(1,0);
                text[0]='T'; text[1]='='; printtemp(10*tapwater,text+2);
                lcd_print8(text);
                break;
	case 14: //meteo
		lcd_cursor(0,0);
                lcd_print8(menulabel[menustate-2]);
      		text[1]='=';
		switch(meteoselect)
			{
//implement also receive of meteo2 type meteostation
#if defined(METEOSENSOR)
			case 0:
                		text[0]='T';
                		printtemp(temperatures[METEOSENSOR],text+2);
				break;
			case 1:
				text[0]='H';
				printhum(humidities[METEOSENSOR],text+2);
				text[6]=text[7]=' ';
				break;
			case 2:
				text[0]='P';
				printpres(pressures[METEOSENSOR],text+2);
				break;
			case 3:
				text[0]='L';
				memset(text+2,' ',6);
				itoa10(lights[METEOSENSOR],text+2);
				break;
#endif
#if defined(METEOSENSOR2)
                        case 4:
                                text[0]='A'; 
                                memset(text+2,' ',6);
                                itoa10(azimuths[METEOSENSOR2],text+2);
                                break;
                        case 5:
                                text[0]='W';
                                memset(text+2,' ',6);
                                printpres(windspeeds[METEOSENSOR2],text+2);
                                break;
			case 6:
				text[0]='R';
                                memset(text+2,' ',6);
                                printpres(rains[METEOSENSOR2],text+2);
                                break;
#endif
			default:
				break;
			}
          	text[8]=0;
		lcd_cursor(1,0);
             	lcd_print8(text);
		break;
	case 7:
	case 6:
	case 5:
		{
		int16_t t;
		if(menustate==7) t=eeprom_read_word(&ee_tempmax);
		if(menustate==6) t=tempsum/tempcount;
		if(menustate==5) t=eeprom_read_word(&ee_tempmin);
		lcd_cursor(0,0);
                lcd_print8(menulabel[menustate-2]);
                lcd_cursor(1,0);
		text[0]='T';
                text[1]=tempchar;
                printtemp(t,text+2);
		text[8]=0;
		lcd_print8(text);
		}
		break;
	case 4:
		lcd_cursor(0,0);
		lcd_print8(menulabel[menustate-2]);
		lcd_cursor(1,0);
		strcpy(text,"HOFF=");
                itoa10(hysteresis_off,text+5);
                lcd_print8(text);
		break;
	case 3:
		lcd_cursor(0,0);
                lcd_print8(menulabel[menustate-2]);
		lcd_cursor(1,0);
		strcpy(text,"HON=");
		itoa10(hysteresis_on,text+4);
		lcd_print8(text);
		break;
	case 2:
		lcd_cursor(0,0);
                lcd_print8(menulabel[menustate-2]);
		lcd_cursor(1,0);
		lcd_print8(activelabel[activestate]);
		break;
	case 1:
		lcd_cursor(0,0);
		lcd_print8("MENU:");
		lcd_cursor(1,0);
		lcd_print8(menulabel[menuselect]);
		break;
	case 0:
	default:
		text[0]='T';
		text[1]=tempchar;
		printtemp(temp,text+2);
		text[8]=0;
		lcd_cursor(0,0);
		lcd_print8(text);

		if(activestate)
			{
			text[0]= label[activestate];
			text[1]=':';
			printtemp(tempset,text+2);
			text[8]=0;
			}
		else strcpy(text,activelabel[activestate]);

		lcd_cursor(1,0);
		lcd_print8(text);
		if(lock_tempset)
			{
			lcd_cursor(1,7);
			lcd_print("L");
			}
	break;

}

//check received messages
{
CODE433 c;
DEVICE device;
SERIAL serial;
serial=receive(&c,&device);
if(serial == (SERIAL)-1) goto nothing_new;
if (device==ACTOR_DEVICE &&  ACTOR_SERIAL == serial) //command for us
	{
	printP(PSTR("command received OK\n"));
	if((c.cmd.instr & 0xf0) == RELAY)
                switch(c.cmd.instr & 0x0f) {
			case RELAY_TIMED: //activate relay for a given time c.cmd.arg2 deciseconds
				if(c.cmd.arg1&1)
					{
					sbi(PORTE,PE3);
					counter2 = ((uint32_t)c.cmd.arg2 *(XTAL/1024/10))>>8;
					}
				break;
			default:
				printP(PSTR("unsupported RELAY actor command\n"));
				break;
		}
	if((c.cmd.instr & 0xf0) == THERMOSTAT) 
		switch(c.cmd.instr & 0x0f) {
			case LOCK_TEMPSET:
				lock_tempset = c.cmd.arg1;
				break;
			case SET_OUTPUT_COUNTER:
				counter_output=c.cmd.arg2;
				eeprom_write_word(&counter_eeprom_output,counter_output);
				break;
			case RESET_HEATER:
				reset_heater();
				break;
			case SET_EFFECTIVE:
				printP(PSTR("SET_EFFECTIVE\n"));
				eff_temp_kind=c.cmd.arg1;
				eff_temp_from=c.cmd.arg2>>8;
				eff_temp_to=c.cmd.arg2&0xff;
				break;
			case SET_REGIME:
				printP(PSTR("SET_REGIME\n"));
				activestate=c.cmd.arg1;
				if(activestate == REMOTE) remotecounter=0;
				break;
			case SET_TEMPERATURE:
				watchdog(1); delay_ms(200); //do not transmit confirmation until server transmission is not over
				printP(PSTR("SET_TEMPERATURE\n"));
				report1a=1;
				switch(c.cmd.arg1) {
				case 1: //at least
					if(c.cmd.arg2 > tempset) tempset=c.cmd.arg2;
					break;
				case 2: //at most
					if(c.cmd.arg2 < tempset) tempset=c.cmd.arg2;
					break;
				default: //just set
					tempset = c.cmd.arg2;
				}
				break;
			case SET_HYSTON:
				printP(PSTR("SET_HYSTON\n"));
				hysteresis_on = c.cmd.arg1;
				break;
			case SET_HYSTOFF:
				printP(PSTR("SET_HYSTOFF\n"));
				hysteresis_off = c.cmd.arg1;
				break;
			case HEATCOOL_CTRL:
				printP(PSTR("HEATCOOL_CTRL\n"));
				if(c.cmd.arg1 & HEATER_ON) heater(1);
				if(c.cmd.arg1 & HEATER_OFF) heater(0);
				if(c.cmd.arg1 & COOLER_ON) cooler(1);
				if(c.cmd.arg1 & COOLER_OFF) cooler(0);
				break;
			case GET_STATUS:
				printP(PSTR("GET_STATUS\n"));
				inquire_status=1;
				break;
			default:	
				printP(PSTR("unknown actor cmd\n"));
				break;
		}
	}

if(device==SENSOR_DEVICE && (c.cmd.instr & 0xf0) == CLOCK)
	{
	switch(c.cmd.instr & 0x0f) {
		case TIME_REPORT:
			printP(PSTR("TIME MARK received: "));
			{
			uint8_t i;
			char z[4];
			for(i=0; i<4; ++i)
                	{
                	bin2octet(z,(c.code>>(3-i))&0xff);
                	z[2]=0;
                	print(z);
                	}
        		printP(PSTR("\n"));
			}
			if(c.time.isday) printP(PSTR("DAY\n")); else printP(PSTR("NIGHT\n"));
			//light = c.time.isday? 0 : (1<<PC7);
			break;
		default:
			printP(PSTR("irrelevant clock report\n"));
			break;
		}
	}

if(device==SENSOR_DEVICE && (c.cmd.instr & 0xf0) == THERMOMETER)
	{
	printP(PSTR("EXTERNAL REPORT RECEIVED\n"));
	switch(c.cmd.instr & 0x0f) {
		case TEMPERATURE_REPORT:
			printP(PSTR("TEMPERATURE_REPORT\n"));
			if(serial>=0 && serial <NTHERMOMETERS)
				{
				temperatures[serial] = c.cmd.arg2;
				temp_timeouts[serial] = 0;
				}
			break;
		case PRESSURE_REPORT:
			printP(PSTR("PRESSURE_REPORT\n"));
			if(serial>=0 && serial <NTHERMOMETERS)
                                {
                                pressures[serial] = c.cmd.arg2;
				//correct to sea level  - approximately for constant temperature 10C
				{
				uint16_t tmp = pressures[serial];
				tmp>>= 5;
				pressures[serial] += tmp;
				}
                                temp_timeouts[serial] = 0;
                                }
			break;
		case HUMIDITY_REPORT:
			printP(PSTR("HUMIDITY_REPORT\n"));
                        if(serial>=0 && serial <NTHERMOMETERS)
                                {
                                humidities[serial] = c.cmd.arg1;
                                temp_timeouts[serial] = 0;
                                }
			break;
		case WIND_BLOW_REPORT:
			printP(PSTR("WIND_BLOW_REPORT\n"));
			break;
		case WIND_REPORT:
			printP(PSTR("WIND_REPORT\n"));
                        if(serial>=0 && serial <NTHERMOMETERS)
                                { //@@@probably not ok, check
                                azimuths[serial] = (360*(uint16_t)c.cmd.arg1)>>8; 
				azimuths[serial] += 0/*mounting offset*/; if(azimuths[serial] > 360) azimuths[serial] -= 360; //mounting offset
				uint32_t tmp=c.cmd.arg2;
				tmp *= 41;
				tmp >>= 10;
                                windspeeds[serial] = (uint16_t) tmp;
                                }
			break;
		case RAIN_REPORT:
				printP(PSTR("RAIN_REPORT\n"));
				rains[serial] +=  (55*c.cmd.arg2)/10;
			break;
		case RAINFALL_REPORT:
				printP(PSTR("RAINFALL_REPORT\n"));
				//@@@not implemented at the moment
			break;
		case LIGHT_REPORT:
			printP(PSTR("LIGHT_REPORT\n"));
                        if(serial>=0 && serial <NTHERMOMETERS)
                                {
                                lights[serial] = c.cmd.arg2;
                                temp_timeouts[serial] = 0;
                                }
			break;
		default:
			printP(PSTR("unknown thermometer report\n"));
			break;
		}
	}
nothing_new:;
}

//set external temperatures to undefined if temp_timeouts exceed threshold
{
uint8_t i;
for(i=0; i<NTHERMOMETERS; ++i)
    if(i!=SENSOR_SERIAL)
	{
	uint16_t t= eeprom_read_word(&temp_timeouts_thr[i]);
	if(temp_timeouts[i] > t && t>0) 
		{
		temperatures[i] = UNDEFINED_TEMP;
		pressures[i] = UNDEFINED_PRESSURE;
		humidities[i] = UNDEFINED_HUMID;
		lights[i] = UNDEFINED_LIGHT;
		}
	}
}

//change from remote state to  heatcool if there was no set regime message in last x passes corresponding to about 30 minutes (32000 passes of about .05 seconds)
if(activestate == REMOTE)
	{
	if(remotecounter>30000)
		{
		activestate= COOLER|HEATER;
		remotecounter=0;
		}
	}

if(clearingcounter>30000)
	{
	//switch off light
	light =0;
        cbi(PORTC,PC7);
	//return to main menu
	menustate=menuselect=0;
	clearingcounter=0;
	}

//switch on/off heating/cooling and corresponding LED and send message about that
if(activestate&HEATER && temp!= UNDEFINED_TEMP) 
		{
		if(heater_on && temp>=tempset+hysteresis_off) heater(0);
		if(!heater_on && temp<=tempset-hysteresis_on) heater(1);
		}

if(activestate&COOLER && temp!= UNDEFINED_TEMP) 
		{
		if(cooler_on && temp<=tempset-hysteresis_off) cooler(0);
		if(!cooler_on && temp>=tempset+hysteresis_on) cooler(1);
		}

if(reportheatcool)
	{
        CODE433 c;
        c.cmd.instr= THERMOMETER|TEMPERATURE_AND_HEATCOOL;
	c.cmd.arg1=reportheatcool;
        c.cmd.arg2=temperatures[SENSOR_SERIAL]; //not the effective temerature
#ifdef debug
	printP(PSTR("transmit from reportheatcool\n"));
#endif
	reportheatcool=0;
        transmit(c.code,TXCOUNT);
	}

if((activestate&REMOTE) == 0) //not in remote-controlled mode, switch off heater/cooler if they should not be active
	{
	if((activestate&HEATER)==0 && heater_on) heater(0);
	if((activestate&COOLER)==0 && cooler_on) cooler(0);
	}

//report temperature; switch off display light, switch off menustate after some time???

if((pass&0x1fff) == 0 || (activestate&REMOTE) && (pass&0x7ff) == 0 || inquire_status|| temperatures[SENSOR_SERIAL]>reported_temp+2||temperatures[SENSOR_SERIAL]<reported_temp-2)
	send_temp_report(tempchar);

//store sensor_counters to eeprom
if((pass&0x7fff) == 0)
	{
	uint8_t i;
	for(i=0; i<NSENSORS; ++i) 
		{
		if(sensors_counters[i] != eeprom_read_dword(&sensors_counters_init[i]))
			eeprom_write_dword(&sensors_counters_init[i],sensors_counters[i]);
		}
	}

//each n-th pass write changed values to eeprom and
if((pass&0x7f)==0 || inquire_status ||report1a ) 
	{
	uint8_t report1=0;
	uint8_t report2=0;
	uint8_t report3=0;
	if(tempset!=eeprom_read_word(&ee_tempset)) {eeprom_write_word(&ee_tempset,tempset); report1=1;}
	if(hysteresis_on!=eeprom_read_byte(&ee_hysteresis_on)) {eeprom_write_byte(&ee_hysteresis_on,hysteresis_on); report2=1;}
	if(hysteresis_off!=eeprom_read_byte(&ee_hysteresis_off)) {eeprom_write_byte(&ee_hysteresis_off,hysteresis_off); report2=1;}
	if(activestate!=eeprom_read_byte(&ee_activestate)) {eeprom_write_byte(&ee_activestate,activestate); report2=1;}
	if(eff_temp_from!=eeprom_read_byte(&ee_eff_temp_from)) {eeprom_write_byte(&ee_eff_temp_from,eff_temp_from); report3=1;}
	if(eff_temp_to!=eeprom_read_byte(&ee_eff_temp_to)) {eeprom_write_byte(&ee_eff_temp_to,eff_temp_to); report3=1;}
	if(eff_temp_kind!=eeprom_read_byte(&ee_eff_temp_kind)) {eeprom_write_byte(&ee_eff_temp_kind,eff_temp_kind); report3=1;}
	if(report1||report2||report3) printP(PSTR("eeprom updated\n"));

	//send messages about user-changed settings 
	if(report3 || inquire_status)
                {
                CODE433 c;
		report3=0;
                c.cmd.instr= THERMOMETER|EFFECTIVE_TEMP;
                c.cmd.arg1=eff_temp_kind;
                c.cmd.arg2 = eff_temp_from<<8|eff_temp_to;
#ifdef debug
        printP(PSTR("transmit from report3\n"));
#endif
                transmit(c.code,TXCOUNT);
                }
	if(report2 || inquire_status)
		{
		CODE433 c;
		report2=0;
		c.cmd.instr= THERMOMETER|REGIME_AND_HYST;
		c.cmd.arg1=activestate;
		c.cmd.arg2 = hysteresis_on<<8 |hysteresis_off;
#ifdef debug
        printP(PSTR("transmit from report2\n"));
#endif
		transmit(c.code,TXCOUNT);
		}
	if(report1a || report1 || inquire_status)
		{
		CODE433 c;
		report1=0;
		report1a=0;
		c.cmd.instr= THERMOMETER|TEMPSET_AND_HEATCOOL;
		c.cmd.arg2 = tempset;
		c.cmd.arg1= (heater_on?HEATER_ON:HEATER_OFF) | (cooler_on?COOLER_ON:COOLER_OFF);
#ifdef debug
        printP(PSTR("transmit from report1\n"));
#endif
		transmit(c.code,TXCOUNT);
		}
	inquire_status=0;
	}
delay_ms(100);
if(pass==0) lcd_init(); //in the case DDR register becomes corruped
}//while

}

